/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.io.orc;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.GnuParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.OptionBuilder;
import org.apache.commons.cli.Options;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hive.ql.io.RecordIdentifier;
import org.apache.hadoop.hive.ql.io.orc.CompressionKind;
import org.apache.hadoop.hive.ql.io.orc.OrcFile;
import org.apache.hadoop.hive.ql.io.orc.OrcInputFormat;
import org.apache.hadoop.hive.ql.io.orc.OrcRecordUpdater;
import org.apache.hadoop.hive.ql.io.orc.OrcStruct;
import org.apache.hadoop.hive.ql.io.orc.Reader;
import org.apache.hadoop.hive.ql.io.orc.RecordReader;
import org.apache.hadoop.hive.ql.io.orc.Writer;
import org.apache.hadoop.hive.serde2.objectinspector.StructField;
import org.apache.hadoop.hive.serde2.objectinspector.StructObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.IntObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.LongObjectInspector;
import org.apache.orc.OrcProto;
import org.apache.orc.StripeInformation;
import org.apache.orc.tools.FileDump;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FixAcidKeyIndex {
    public static final Logger LOG = LoggerFactory.getLogger(FixAcidKeyIndex.class);
    public static final String DEFAULT_BACKUP_PATH = System.getProperty("java.io.tmpdir");
    private static final Charset UTF8 = Charset.forName("UTF-8");
    private static final CharsetDecoder utf8Decoder = UTF8.newDecoder();

    public static void main(String[] args) throws Exception {
        Configuration conf = new Configuration();
        Options opts = FixAcidKeyIndex.createOptions();
        CommandLine cli = new GnuParser().parse(opts, args);
        if (cli.hasOption('h')) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("fixacidkeyindex", opts);
            return;
        }
        Properties configProps = cli.getOptionProperties("hiveconf");
        for (String key : configProps.stringPropertyNames()) {
            conf.set(key, configProps.getProperty(key));
        }
        String backupPath = DEFAULT_BACKUP_PATH;
        if (cli.hasOption("backup-path")) {
            backupPath = cli.getOptionValue("backup-path");
        }
        boolean checkOnly = cli.hasOption("check-only");
        boolean recover = cli.hasOption("recover");
        String[] files = cli.getArgs();
        if (files.length == 0) {
            System.err.println("Error : ORC files are not specified");
            return;
        }
        ArrayList<String> filesInPath = new ArrayList<String>();
        for (String filename : files) {
            Path path = new Path(filename);
            filesInPath.addAll(FixAcidKeyIndex.getAllFilesInPath(path, conf));
        }
        if (checkOnly) {
            FixAcidKeyIndex.checkFiles(conf, filesInPath);
        } else if (recover) {
            FixAcidKeyIndex.recoverFiles(conf, filesInPath, backupPath);
        } else {
            System.err.println("check-only or recover option must be specified");
        }
    }

    static void recoverFiles(Configuration conf, List<String> fileList, String backup) {
        for (String fileName : fileList) {
            try {
                Path filePath = new Path(fileName);
                FixAcidKeyIndex.recoverFile(conf, filePath, backup);
            }
            catch (Exception err) {
                System.err.println("ERROR recovering " + fileName);
                err.printStackTrace(System.err);
            }
        }
    }

    static void checkFiles(Configuration conf, List<String> fileList) {
        for (String fileName : fileList) {
            try {
                Path filePath = new Path(fileName);
                FixAcidKeyIndex.checkFile(conf, filePath);
            }
            catch (Exception err) {
                System.err.println("ERROR checking " + fileName);
                err.printStackTrace(System.err);
            }
        }
    }

    static void checkFile(Configuration conf, Path inputPath) throws IOException {
        FileSystem fs = inputPath.getFileSystem(conf);
        try (Reader reader = OrcFile.createReader(fs, inputPath);){
            if (OrcInputFormat.isOriginal(reader)) {
                System.out.println(String.valueOf(inputPath) + " is not an acid file");
                return;
            }
        }
        AcidKeyIndexValidationResult validationResult = FixAcidKeyIndex.validate(conf, inputPath);
        boolean validIndex = validationResult.isValid;
        System.out.println("Checking " + String.valueOf(inputPath) + " - acid key index is " + (validIndex ? "valid" : "invalid"));
    }

    public static AcidKeyIndexValidationResult validate(Configuration conf, Path inputPath) throws IOException {
        AcidKeyIndexValidationResult result = new AcidKeyIndexValidationResult();
        FileSystem fs = inputPath.getFileSystem(conf);
        try (Reader reader = OrcFile.createReader(fs, inputPath);
             RecordReader rr = reader.rows();){
            List stripes = reader.getStripes();
            RecordIdentifier[] keyIndex = OrcRecordUpdater.parseKeyIndex(reader);
            if (keyIndex == null) {
                result.isValid = false;
            }
            StructObjectInspector soi = (StructObjectInspector)reader.getObjectInspector();
            List structFields = soi.getAllStructFieldRefs();
            StructField transactionField = (StructField)structFields.get(1);
            LongObjectInspector transactionOI = (LongObjectInspector)transactionField.getFieldObjectInspector();
            StructField bucketField = (StructField)structFields.get(2);
            IntObjectInspector bucketOI = (IntObjectInspector)bucketField.getFieldObjectInspector();
            StructField rowIdField = (StructField)structFields.get(3);
            LongObjectInspector rowIdOI = (LongObjectInspector)rowIdField.getFieldObjectInspector();
            long rowsProcessed = 0L;
            for (int i = 0; i < stripes.size(); ++i) {
                rr.seekToRow((rowsProcessed += ((StripeInformation)stripes.get(i)).getNumberOfRows()) - 1L);
                OrcStruct row = (OrcStruct)rr.next(null);
                long lastTransaction = transactionOI.get(soi.getStructFieldData((Object)row, transactionField));
                int lastBucket = bucketOI.get(soi.getStructFieldData((Object)row, bucketField));
                long lastRowId = rowIdOI.get(soi.getStructFieldData((Object)row, rowIdField));
                RecordIdentifier recordIdentifier = new RecordIdentifier(lastTransaction, lastBucket, lastRowId);
                result.recordIdentifiers.add(recordIdentifier);
                if (!result.isValid || stripes.size() == keyIndex.length && keyIndex[i] != null && recordIdentifier.compareTo(keyIndex[i]) == 0) continue;
                result.isValid = false;
            }
        }
        return result;
    }

    static void recoverFile(Configuration conf, Path inputPath, String backup) throws IOException {
        FileSystem fs = inputPath.getFileSystem(conf);
        Path recoveredPath = FixAcidKeyIndex.getRecoveryFile(inputPath);
        try (Reader reader = OrcFile.createReader(fs, inputPath);){
            if (OrcInputFormat.isOriginal(reader)) {
                System.out.println(String.valueOf(inputPath) + " is not an acid file. No need to recover.");
                return;
            }
            AcidKeyIndexValidationResult validationResult = FixAcidKeyIndex.validate(conf, inputPath);
            if (validationResult.isValid) {
                System.out.println(String.valueOf(inputPath) + " has a valid acid key index. No need to recover.");
                return;
            }
            System.out.println("Recovering " + String.valueOf(inputPath));
            try {
                fs.delete(recoveredPath, false);
            }
            catch (FileNotFoundException fileNotFoundException) {
                // empty catch block
            }
            OrcFile.WriterOptions writerOptions = OrcFile.writerOptions(conf).compress(reader.getCompression()).version(reader.getFileVersion()).rowIndexStride(reader.getRowIndexStride()).inspector(reader.getObjectInspector());
            if (reader.getCompression() != CompressionKind.NONE) {
                writerOptions.bufferSize(reader.getCompressionSize()).enforceBufferSize();
            }
            try (Writer writer = OrcFile.createWriter(recoveredPath, writerOptions);){
                List stripes = reader.getStripes();
                List stripeStats = reader.getOrcProtoStripeStatistics();
                try (FSDataInputStream inputStream = fs.open(inputPath);){
                    for (int idx = 0; idx < stripes.size(); ++idx) {
                        StripeInformation stripe = (StripeInformation)stripes.get(idx);
                        int stripeLength = (int)stripe.getLength();
                        byte[] buffer = new byte[stripeLength];
                        inputStream.readFully(stripe.getOffset(), buffer, 0, stripeLength);
                        writer.appendStripe(buffer, 0, buffer.length, stripe, (OrcProto.StripeStatistics)stripeStats.get(idx));
                    }
                }
                for (String metadataKey : reader.getMetadataKeys()) {
                    if (metadataKey.equals("hive.acid.key.index")) continue;
                    writer.addUserMetadata(metadataKey, reader.getMetadataValue(metadataKey));
                }
                StringBuilder sb = new StringBuilder();
                validationResult.recordIdentifiers.stream().forEach(ri -> sb.append(ri.getWriteId()).append(",").append(ri.getBucketProperty()).append(",").append(ri.getRowId()).append(";"));
                writer.addUserMetadata("hive.acid.key.index", UTF8.encode(sb.toString()));
            }
        }
        AcidKeyIndexValidationResult fileFixed = FixAcidKeyIndex.validate(conf, recoveredPath);
        if (fileFixed.isValid) {
            String scheme = inputPath.toUri().getScheme();
            String authority = inputPath.toUri().getAuthority();
            String filePath = inputPath.toUri().getPath();
            Path backupDataPath = backup.equals(DEFAULT_BACKUP_PATH) ? new Path(scheme, authority, DEFAULT_BACKUP_PATH + filePath) : Path.mergePaths((Path)new Path(backup), (Path)inputPath);
            FixAcidKeyIndex.moveFiles(fs, inputPath, backupDataPath);
            FixAcidKeyIndex.moveFiles(fs, recoveredPath, inputPath);
            System.out.println("Fixed acid key index for " + String.valueOf(inputPath));
        } else {
            System.out.println("Unable to fix acid key index for " + String.valueOf(inputPath));
        }
    }

    private static void moveFiles(FileSystem fs, Path src, Path dest) throws IOException {
        try {
            if (!fs.exists(dest.getParent())) {
                fs.mkdirs(dest.getParent());
            }
            fs.delete(dest, false);
            if (!fs.rename(src, dest)) {
                throw new IOException("Unable to move " + String.valueOf(src) + " to " + String.valueOf(dest));
            }
            System.err.println("Moved " + String.valueOf(src) + " to " + String.valueOf(dest));
        }
        catch (Exception e) {
            throw new IOException("Unable to move " + String.valueOf(src) + " to " + String.valueOf(dest), e);
        }
    }

    static String getKeyIndexAsString(Reader reader) {
        try {
            ByteBuffer val = reader.getMetadataValue("hive.acid.key.index").duplicate();
            return utf8Decoder.decode(val).toString();
        }
        catch (CharacterCodingException e) {
            throw new IllegalArgumentException("Bad string encoding for hive.acid.key.index", e);
        }
    }

    static Path getRecoveryFile(Path corruptPath) {
        return new Path(corruptPath.getParent(), corruptPath.getName() + ".fixacidindex");
    }

    static Options createOptions() {
        Options result = new Options();
        OptionBuilder.withLongOpt((String)"check-only");
        OptionBuilder.withDescription((String)"Check acid orc file for valid acid key index and exit without fixing");
        result.addOption(OptionBuilder.create((char)'c'));
        OptionBuilder.withLongOpt((String)"recover");
        OptionBuilder.withDescription((String)"Fix the acid key index for acid orc file if it requires fixing");
        result.addOption(OptionBuilder.create((char)'r'));
        OptionBuilder.withLongOpt((String)"backup-path");
        OptionBuilder.withDescription((String)"specify a backup path to store the corrupted files (default: /tmp)");
        OptionBuilder.hasArg();
        result.addOption(OptionBuilder.create());
        OptionBuilder.withValueSeparator();
        OptionBuilder.hasArgs((int)2);
        OptionBuilder.withArgName((String)"property=value");
        OptionBuilder.withLongOpt((String)"hiveconf");
        OptionBuilder.withDescription((String)"Use value for given property");
        result.addOption(OptionBuilder.create());
        OptionBuilder.withLongOpt((String)"help");
        OptionBuilder.withDescription((String)"print help message");
        result.addOption(OptionBuilder.create((char)'h'));
        return result;
    }

    public static Collection<String> getAllFilesInPath(Path path, Configuration conf) throws IOException {
        ArrayList<String> filesInPath = new ArrayList<String>();
        FileSystem fs = path.getFileSystem(conf);
        FileStatus fileStatus = fs.getFileStatus(path);
        if (fileStatus.isDir()) {
            FileStatus[] fileStatuses;
            for (FileStatus fileInPath : fileStatuses = fs.listStatus(path, FileDump.HIDDEN_AND_SIDE_FILE_FILTER)) {
                if (fileInPath.isDir()) {
                    filesInPath.addAll(FixAcidKeyIndex.getAllFilesInPath(fileInPath.getPath(), conf));
                    continue;
                }
                filesInPath.add(fileInPath.getPath().toString());
            }
        } else {
            filesInPath.add(path.toString());
        }
        return filesInPath;
    }

    private static class AcidKeyIndexValidationResult {
        private boolean isValid = true;
        private List<RecordIdentifier> recordIdentifiers = new LinkedList<RecordIdentifier>();

        private AcidKeyIndexValidationResult() {
        }
    }
}

