/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import com.yammer.metrics.core.Timer;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.attribute.FileTime;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Optional;
import java.util.OptionalLong;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.kafka.common.InvalidRecordException;
import org.apache.kafka.common.errors.CorruptRecordException;
import org.apache.kafka.common.record.FileLogInputStream;
import org.apache.kafka.common.record.FileRecords;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.RecordBatch;
import org.apache.kafka.common.record.Records;
import org.apache.kafka.common.utils.BufferSupplier;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.metrics.KafkaMetricsGroup;
import org.apache.kafka.storage.internals.epoch.LeaderEpochFileCache;
import org.apache.kafka.storage.internals.log.AbortedTxn;
import org.apache.kafka.storage.internals.log.AppendOrigin;
import org.apache.kafka.storage.internals.log.CompletedTxn;
import org.apache.kafka.storage.internals.log.FetchDataInfo;
import org.apache.kafka.storage.internals.log.LazyIndex;
import org.apache.kafka.storage.internals.log.LogConfig;
import org.apache.kafka.storage.internals.log.LogFileUtils;
import org.apache.kafka.storage.internals.log.LogOffsetMetadata;
import org.apache.kafka.storage.internals.log.LogSegmentOffsetOverflowException;
import org.apache.kafka.storage.internals.log.OffsetIndex;
import org.apache.kafka.storage.internals.log.OffsetPosition;
import org.apache.kafka.storage.internals.log.ProducerAppendInfo;
import org.apache.kafka.storage.internals.log.ProducerStateManager;
import org.apache.kafka.storage.internals.log.RollParams;
import org.apache.kafka.storage.internals.log.StorageAction;
import org.apache.kafka.storage.internals.log.TimeIndex;
import org.apache.kafka.storage.internals.log.TimestampOffset;
import org.apache.kafka.storage.internals.log.TransactionIndex;
import org.apache.kafka.storage.internals.log.TxnIndexSearchResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.event.Level;

public class LogSegment
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogSegment.class);
    private static final Timer LOG_FLUSH_TIMER;
    private static final String FUTURE_DIR_SUFFIX = "-future";
    private static final Pattern FUTURE_DIR_PATTERN;
    private final FileRecords log;
    private final LazyIndex<OffsetIndex> lazyOffsetIndex;
    private final LazyIndex<TimeIndex> lazyTimeIndex;
    private final TransactionIndex txnIndex;
    private final long baseOffset;
    private final int indexIntervalBytes;
    private final long rollJitterMs;
    private final Time time;
    private volatile OptionalLong rollingBasedTimestamp = OptionalLong.empty();
    private volatile TimestampOffset maxTimestampAndOffsetSoFar = TimestampOffset.UNKNOWN;
    private final Object maxTimestampAndOffsetLock = new Object();
    private long created;
    private int bytesSinceLastIndexEntry = 0;

    public LogSegment(FileRecords log, LazyIndex<OffsetIndex> lazyOffsetIndex, LazyIndex<TimeIndex> lazyTimeIndex, TransactionIndex txnIndex, long baseOffset, int indexIntervalBytes, long rollJitterMs, Time time) {
        this.log = log;
        this.lazyOffsetIndex = lazyOffsetIndex;
        this.lazyTimeIndex = lazyTimeIndex;
        this.txnIndex = txnIndex;
        this.baseOffset = baseOffset;
        this.indexIntervalBytes = indexIntervalBytes;
        this.rollJitterMs = rollJitterMs;
        this.time = time;
        this.created = time.milliseconds();
    }

    public OffsetIndex offsetIndex() throws IOException {
        return this.lazyOffsetIndex.get();
    }

    public File offsetIndexFile() {
        return this.lazyOffsetIndex.file();
    }

    public TimeIndex timeIndex() throws IOException {
        return this.lazyTimeIndex.get();
    }

    public File timeIndexFile() {
        return this.lazyTimeIndex.file();
    }

    public long baseOffset() {
        return this.baseOffset;
    }

    public FileRecords log() {
        return this.log;
    }

    public long rollJitterMs() {
        return this.rollJitterMs;
    }

    public TransactionIndex txnIndex() {
        return this.txnIndex;
    }

    public boolean shouldRoll(RollParams rollParams) throws IOException {
        boolean reachedRollMs = this.timeWaitedForRoll(rollParams.now, rollParams.maxTimestampInMessages) > rollParams.maxSegmentMs - this.rollJitterMs;
        int size = this.size();
        return size > rollParams.maxSegmentBytes - rollParams.messagesSize || size > 0 && reachedRollMs || this.offsetIndex().isFull() || this.timeIndex().isFull() || !this.canConvertToRelativeOffset(rollParams.maxOffsetInMessages);
    }

    public void resizeIndexes(int size) throws IOException {
        this.offsetIndex().resize(size);
        this.timeIndex().resize(size);
    }

    public void sanityCheck(boolean timeIndexFileNewlyCreated) throws IOException {
        if (this.offsetIndexFile().exists()) {
            if (timeIndexFileNewlyCreated) {
                this.timeIndex().resize(0);
            }
        } else {
            throw new NoSuchFileException("Offset index file " + this.offsetIndexFile().getAbsolutePath() + " does not exist");
        }
        this.txnIndex.sanityCheck();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public TimestampOffset readMaxTimestampAndOffsetSoFar() throws IOException {
        if (this.maxTimestampAndOffsetSoFar == TimestampOffset.UNKNOWN) {
            Object object = this.maxTimestampAndOffsetLock;
            synchronized (object) {
                if (this.maxTimestampAndOffsetSoFar == TimestampOffset.UNKNOWN) {
                    this.maxTimestampAndOffsetSoFar = this.timeIndex().lastEntry();
                }
            }
        }
        return this.maxTimestampAndOffsetSoFar;
    }

    public long maxTimestampSoFar() throws IOException {
        return this.readMaxTimestampAndOffsetSoFar().timestamp;
    }

    private long shallowOffsetOfMaxTimestampSoFar() throws IOException {
        return this.readMaxTimestampAndOffsetSoFar().offset;
    }

    public int size() {
        return this.log.sizeInBytes();
    }

    private boolean canConvertToRelativeOffset(long offset) throws IOException {
        return this.offsetIndex().canAppendOffset(offset);
    }

    public void append(long largestOffset, MemoryRecords records) throws IOException {
        if (records.sizeInBytes() > 0) {
            LOGGER.trace("Inserting {} bytes at end offset {} at position {}", new Object[]{records.sizeInBytes(), largestOffset, this.log.sizeInBytes()});
            int physicalPosition = this.log.sizeInBytes();
            this.ensureOffsetInRange(largestOffset);
            long appendedBytes = this.log.append(records);
            LOGGER.trace("Appended {} to {} at end offset {}", new Object[]{appendedBytes, this.log.file(), largestOffset});
            for (RecordBatch batch : records.batches()) {
                long batchMaxTimestamp = batch.maxTimestamp();
                long batchLastOffset = batch.lastOffset();
                if (batchMaxTimestamp > this.maxTimestampSoFar()) {
                    this.maxTimestampAndOffsetSoFar = new TimestampOffset(batchMaxTimestamp, batchLastOffset);
                }
                if (this.bytesSinceLastIndexEntry > this.indexIntervalBytes) {
                    this.offsetIndex().append(batchLastOffset, physicalPosition);
                    this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar());
                    this.bytesSinceLastIndexEntry = 0;
                }
                int sizeInBytes = batch.sizeInBytes();
                physicalPosition += sizeInBytes;
                this.bytesSinceLastIndexEntry += sizeInBytes;
            }
        }
    }

    private void ensureOffsetInRange(long offset) throws IOException {
        if (!this.canConvertToRelativeOffset(offset)) {
            throw new LogSegmentOffsetOverflowException(this, offset);
        }
    }

    private int appendChunkFromFile(FileRecords records, int position, BufferSupplier bufferSupplier) throws IOException {
        FileLogInputStream.FileChannelRecordBatch batch;
        int bytesToAppend = 0;
        long maxOffset = Long.MIN_VALUE;
        ByteBuffer readBuffer = bufferSupplier.get(0x100000);
        Iterator<FileLogInputStream.FileChannelRecordBatch> nextBatches = records.batchesFrom(position).iterator();
        while ((batch = this.nextAppendableBatch(nextBatches, readBuffer, bytesToAppend)) != null) {
            maxOffset = batch.lastOffset();
            bytesToAppend += batch.sizeInBytes();
        }
        if (bytesToAppend > 0) {
            if (readBuffer.capacity() < bytesToAppend) {
                readBuffer = bufferSupplier.get(bytesToAppend);
            }
            readBuffer.limit(bytesToAppend);
            records.readInto(readBuffer, position);
            this.append(maxOffset, MemoryRecords.readableRecords((ByteBuffer)readBuffer));
        }
        bufferSupplier.release(readBuffer);
        return bytesToAppend;
    }

    private FileLogInputStream.FileChannelRecordBatch nextAppendableBatch(Iterator<FileLogInputStream.FileChannelRecordBatch> recordBatches, ByteBuffer readBuffer, int bytesToAppend) throws IOException {
        FileLogInputStream.FileChannelRecordBatch batch;
        if (recordBatches.hasNext() && this.canConvertToRelativeOffset((batch = recordBatches.next()).lastOffset()) && (bytesToAppend == 0 || bytesToAppend + batch.sizeInBytes() < readBuffer.capacity())) {
            return batch;
        }
        return null;
    }

    public int appendFromFile(FileRecords records, int start) throws IOException {
        int position;
        int bytesAppended;
        BufferSupplier.GrowableBufferSupplier bufferSupplier = new BufferSupplier.GrowableBufferSupplier();
        for (position = start; position < start + records.sizeInBytes(); position += bytesAppended) {
            bytesAppended = this.appendChunkFromFile(records, position, (BufferSupplier)bufferSupplier);
            if (bytesAppended != 0) continue;
            return position - start;
        }
        return position - start;
    }

    public void updateTxnIndex(CompletedTxn completedTxn, long lastStableOffset) throws IOException {
        if (completedTxn.isAborted) {
            LOGGER.trace("Writing aborted transaction {} to transaction index, last stable offset is {}", (Object)completedTxn, (Object)lastStableOffset);
            this.txnIndex.append(new AbortedTxn(completedTxn, lastStableOffset));
        }
    }

    private void updateProducerState(ProducerStateManager producerStateManager, RecordBatch batch) throws IOException {
        if (batch.hasProducerId()) {
            long producerId = batch.producerId();
            ProducerAppendInfo appendInfo = producerStateManager.prepareUpdate(producerId, AppendOrigin.REPLICATION);
            Optional<CompletedTxn> maybeCompletedTxn = appendInfo.append(batch, Optional.empty());
            producerStateManager.update(appendInfo);
            if (maybeCompletedTxn.isPresent()) {
                CompletedTxn completedTxn = maybeCompletedTxn.get();
                long lastStableOffset = producerStateManager.lastStableOffset(completedTxn);
                this.updateTxnIndex(completedTxn, lastStableOffset);
                producerStateManager.completeTxn(completedTxn);
            }
        }
        producerStateManager.updateMapEndOffset(batch.lastOffset() + 1L);
    }

    public FileRecords.LogOffsetPosition translateOffset(long offset) throws IOException {
        return this.translateOffset(offset, 0);
    }

    FileRecords.LogOffsetPosition translateOffset(long offset, int startingFilePosition) throws IOException {
        OffsetPosition mapping = this.offsetIndex().lookup(offset);
        return this.log.searchForOffsetWithSize(offset, Math.max(mapping.position, startingFilePosition));
    }

    public FetchDataInfo read(long startOffset, int maxSize) throws IOException {
        return this.read(startOffset, maxSize, this.size());
    }

    public FetchDataInfo read(long startOffset, int maxSize, long maxPosition) throws IOException {
        return this.read(startOffset, maxSize, Optional.of(maxPosition), false);
    }

    public FetchDataInfo read(long startOffset, int maxSize, Optional<Long> maxPositionOpt, boolean minOneMessage) throws IOException {
        if (maxSize < 0) {
            throw new IllegalArgumentException("Invalid max size " + maxSize + " for log read from segment " + String.valueOf(this.log));
        }
        FileRecords.LogOffsetPosition startOffsetAndSize = this.translateOffset(startOffset);
        if (startOffsetAndSize == null) {
            return null;
        }
        int startPosition = startOffsetAndSize.position;
        LogOffsetMetadata offsetMetadata = new LogOffsetMetadata(startOffsetAndSize.offset, this.baseOffset, startPosition);
        int adjustedMaxSize = maxSize;
        if (minOneMessage) {
            adjustedMaxSize = Math.max(maxSize, startOffsetAndSize.size);
        }
        if (adjustedMaxSize == 0 || !maxPositionOpt.isPresent()) {
            return new FetchDataInfo(offsetMetadata, (Records)MemoryRecords.EMPTY);
        }
        int fetchSize = Math.min((int)(maxPositionOpt.get() - (long)startPosition), adjustedMaxSize);
        return new FetchDataInfo(offsetMetadata, (Records)this.log.slice(startPosition, fetchSize), adjustedMaxSize < startOffsetAndSize.size, Optional.empty());
    }

    public OptionalLong fetchUpperBoundOffset(OffsetPosition startOffsetPosition, int fetchSize) throws IOException {
        return this.offsetIndex().fetchUpperBoundOffset(startOffsetPosition, fetchSize).map(offsetPosition -> OptionalLong.of(offsetPosition.offset)).orElseGet(OptionalLong::empty);
    }

    public int recover(ProducerStateManager producerStateManager, LeaderEpochFileCache leaderEpochCache) throws IOException {
        this.offsetIndex().reset();
        this.timeIndex().reset();
        this.txnIndex.reset();
        int validBytes = 0;
        int lastIndexEntry = 0;
        this.maxTimestampAndOffsetSoFar = TimestampOffset.UNKNOWN;
        try {
            for (RecordBatch batch : this.log.batches()) {
                batch.ensureValid();
                this.ensureOffsetInRange(batch.lastOffset());
                if (batch.maxTimestamp() > this.maxTimestampSoFar()) {
                    this.maxTimestampAndOffsetSoFar = new TimestampOffset(batch.maxTimestamp(), batch.lastOffset());
                }
                if (validBytes - lastIndexEntry > this.indexIntervalBytes) {
                    this.offsetIndex().append(batch.lastOffset(), validBytes);
                    this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar());
                    lastIndexEntry = validBytes;
                }
                validBytes += batch.sizeInBytes();
                if (batch.magic() < 2) continue;
                if (batch.partitionLeaderEpoch() >= 0 && (leaderEpochCache.latestEpoch().isEmpty() || batch.partitionLeaderEpoch() > leaderEpochCache.latestEpoch().getAsInt())) {
                    leaderEpochCache.assign(batch.partitionLeaderEpoch(), batch.baseOffset());
                }
                this.updateProducerState(producerStateManager, batch);
            }
        }
        catch (InvalidRecordException | CorruptRecordException e) {
            LOGGER.warn("Found invalid messages in log segment {} at byte offset {}.", new Object[]{this.log.file().getAbsolutePath(), validBytes, e});
        }
        int truncated = this.log.sizeInBytes() - validBytes;
        if (truncated > 0) {
            LOGGER.debug("Truncated {} invalid bytes at the end of segment {} during recovery", (Object)truncated, (Object)this.log.file().getAbsolutePath());
        }
        this.log.truncateTo(validBytes);
        this.offsetIndex().trimToValidSize();
        this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true);
        this.timeIndex().trimToValidSize();
        return truncated;
    }

    public boolean hasOverflow() throws IOException {
        long nextOffset = this.readNextOffset();
        return nextOffset > this.baseOffset && !this.canConvertToRelativeOffset(nextOffset - 1L);
    }

    public TxnIndexSearchResult collectAbortedTxns(long fetchOffset, long upperBoundOffset) {
        return this.txnIndex.collectAbortedTxns(fetchOffset, upperBoundOffset);
    }

    public String toString() {
        return "LogSegment(baseOffset=" + this.baseOffset + ", size=" + this.size() + ", lastModifiedTime=" + this.lastModified() + ", largestRecordTimestamp=" + this.maxTimestampAndOffsetSoFar.timestamp + ")";
    }

    public int truncateTo(long offset) throws IOException {
        FileRecords.LogOffsetPosition mapping = this.translateOffset(offset);
        OffsetIndex offsetIndex = this.offsetIndex();
        TimeIndex timeIndex = this.timeIndex();
        offsetIndex.truncateTo(offset);
        timeIndex.truncateTo(offset);
        this.txnIndex.truncateTo(offset);
        offsetIndex.resize(offsetIndex.maxIndexSize());
        timeIndex.resize(timeIndex.maxIndexSize());
        int bytesTruncated = mapping == null ? 0 : this.log.truncateTo(mapping.position);
        if (this.log.sizeInBytes() == 0) {
            this.created = this.time.milliseconds();
            this.rollingBasedTimestamp = OptionalLong.empty();
        }
        this.bytesSinceLastIndexEntry = 0;
        if (this.maxTimestampSoFar() >= 0L) {
            this.maxTimestampAndOffsetSoFar = this.readLargestTimestamp();
        }
        return bytesTruncated;
    }

    private TimestampOffset readLargestTimestamp() throws IOException {
        TimestampOffset lastTimeIndexEntry = this.timeIndex().lastEntry();
        OffsetPosition offsetPosition = this.offsetIndex().lookup(lastTimeIndexEntry.offset);
        FileRecords.TimestampAndOffset maxTimestampOffsetAfterLastEntry = this.log.largestTimestampAfter(offsetPosition.position);
        if (maxTimestampOffsetAfterLastEntry.timestamp > lastTimeIndexEntry.timestamp) {
            return new TimestampOffset(maxTimestampOffsetAfterLastEntry.timestamp, maxTimestampOffsetAfterLastEntry.offset);
        }
        return lastTimeIndexEntry;
    }

    public long readNextOffset() throws IOException {
        FetchDataInfo fetchData = this.read(this.offsetIndex().lastOffset(), this.log.sizeInBytes());
        if (fetchData == null) {
            return this.baseOffset;
        }
        return fetchData.records.lastBatch().map(batch -> batch.nextOffset()).orElse(this.baseOffset);
    }

    public void flush() throws IOException {
        try {
            LOG_FLUSH_TIMER.time((Callable)new Callable<Void>(){

                @Override
                public Void call() throws IOException {
                    LogSegment.this.log.flush();
                    LogSegment.this.offsetIndex().flush();
                    LogSegment.this.timeIndex().flush();
                    LogSegment.this.txnIndex.flush();
                    return null;
                }
            });
        }
        catch (Exception e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            if (e instanceof RuntimeException) {
                throw (RuntimeException)e;
            }
            throw new IllegalStateException("Unexpected exception thrown: " + String.valueOf(e), e);
        }
    }

    void updateParentDir(File dir) {
        this.log.updateParentDir(dir);
        this.lazyOffsetIndex.updateParentDir(dir);
        this.lazyTimeIndex.updateParentDir(dir);
        this.txnIndex.updateParentDir(dir);
    }

    public void changeFileSuffixes(String oldSuffix, String newSuffix) throws IOException {
        this.log.renameTo(new File(Utils.replaceSuffix((String)this.log.file().getPath(), (String)oldSuffix, (String)newSuffix)));
        this.lazyOffsetIndex.renameTo(new File(Utils.replaceSuffix((String)this.offsetIndexFile().getPath(), (String)oldSuffix, (String)newSuffix)));
        this.lazyTimeIndex.renameTo(new File(Utils.replaceSuffix((String)this.timeIndexFile().getPath(), (String)oldSuffix, (String)newSuffix)));
        this.txnIndex.renameTo(new File(Utils.replaceSuffix((String)this.txnIndex.file().getPath(), (String)oldSuffix, (String)newSuffix)));
    }

    public boolean hasSuffix(String suffix) {
        return this.log.file().getName().endsWith(suffix) && this.offsetIndexFile().getName().endsWith(suffix) && this.timeIndexFile().getName().endsWith(suffix) && this.txnIndex.file().getName().endsWith(suffix);
    }

    public void onBecomeInactiveSegment() throws IOException {
        this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true);
        this.offsetIndex().trimToValidSize();
        this.timeIndex().trimToValidSize();
        this.log.trim();
    }

    private void loadFirstBatchTimestamp() {
        Iterator iter;
        if (!this.rollingBasedTimestamp.isPresent() && (iter = this.log.batches().iterator()).hasNext()) {
            this.rollingBasedTimestamp = OptionalLong.of(((FileLogInputStream.FileChannelRecordBatch)iter.next()).maxTimestamp());
        }
    }

    public long timeWaitedForRoll(long now, long messageTimestamp) {
        this.loadFirstBatchTimestamp();
        long ts = this.rollingBasedTimestamp.orElse(-1L);
        if (ts >= 0L) {
            return messageTimestamp - ts;
        }
        return now - this.created;
    }

    public long getFirstBatchTimestamp() {
        this.loadFirstBatchTimestamp();
        OptionalLong timestamp = this.rollingBasedTimestamp;
        if (timestamp.isPresent() && timestamp.getAsLong() >= 0L) {
            return timestamp.getAsLong();
        }
        return Long.MAX_VALUE;
    }

    public Optional<FileRecords.TimestampAndOffset> findOffsetByTimestamp(long timestampMs, long startingOffset) throws IOException {
        TimestampOffset timestampOffset = this.timeIndex().lookup(timestampMs);
        int position = this.offsetIndex().lookup((long)Math.max((long)timestampOffset.offset, (long)startingOffset)).position;
        return Optional.ofNullable(this.log.searchForTimestamp(timestampMs, position, startingOffset));
    }

    @Override
    public void close() throws IOException {
        if (this.maxTimestampAndOffsetSoFar != TimestampOffset.UNKNOWN) {
            Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"maybeAppend", () -> this.timeIndex().maybeAppend(this.maxTimestampSoFar(), this.shallowOffsetOfMaxTimestampSoFar(), true));
        }
        Utils.closeAll((Closeable[])new Closeable[]{this.lazyOffsetIndex, this.lazyTimeIndex, this.log, this.txnIndex});
    }

    void closeHandlers() {
        Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"offsetIndex", () -> this.lazyOffsetIndex.closeHandler());
        Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"timeIndex", () -> this.lazyTimeIndex.closeHandler());
        Utils.swallow((Logger)LOGGER, (Level)Level.WARN, (String)"log", () -> this.log.closeHandlers());
        Utils.closeQuietly((AutoCloseable)this.txnIndex, (String)"txnIndex", (Logger)LOGGER);
    }

    public void deleteIfExists() throws IOException {
        try {
            Utils.tryAll(Arrays.asList(() -> this.deleteTypeIfExists(() -> this.log.deleteIfExists(), "log", this.log.file(), true), () -> this.deleteTypeIfExists(() -> this.lazyOffsetIndex.deleteIfExists(), "offset index", this.offsetIndexFile(), true), () -> this.deleteTypeIfExists(() -> this.lazyTimeIndex.deleteIfExists(), "time index", this.timeIndexFile(), true), () -> this.deleteTypeIfExists(() -> this.txnIndex.deleteIfExists(), "transaction index", this.txnIndex.file(), false)));
        }
        catch (Throwable t) {
            if (t instanceof IOException) {
                throw (IOException)t;
            }
            if (t instanceof Error) {
                throw (Error)t;
            }
            if (t instanceof RuntimeException) {
                throw (RuntimeException)t;
            }
            throw new IllegalStateException("Unexpected exception: " + t.getMessage(), t);
        }
    }

    private Void deleteTypeIfExists(StorageAction<Boolean, IOException> delete, String fileType, File file, boolean logIfMissing) throws IOException {
        try {
            if (delete.execute().booleanValue()) {
                LOGGER.info("Deleted {} {}.", (Object)fileType, (Object)file.getAbsolutePath());
            } else {
                String topicPartitionAbsolutePath;
                File fallbackFile;
                if (logIfMissing) {
                    LOGGER.info("Failed to delete {} {} because it does not exist.", (Object)fileType, (Object)file.getAbsolutePath());
                }
                if (file.getParent() == null) {
                    return null;
                }
                Matcher dirMatcher = FUTURE_DIR_PATTERN.matcher(file.getParent());
                if (dirMatcher.matches() && (fallbackFile = new File(topicPartitionAbsolutePath = dirMatcher.group(1) + "-" + dirMatcher.group(2), file.getName())).exists() && file.getName().endsWith(".deleted") && fallbackFile.delete()) {
                    LOGGER.info("Fallback to delete {} {}.", (Object)fileType, (Object)fallbackFile.getAbsolutePath());
                }
            }
            return null;
        }
        catch (IOException e) {
            throw new IOException("Delete of " + fileType + " " + file.getAbsolutePath() + " failed.", e);
        }
    }

    public boolean deleted() {
        return !this.log.file().exists() && !this.offsetIndexFile().exists() && !this.timeIndexFile().exists() && !this.txnIndex.file().exists();
    }

    public long lastModified() {
        return this.log.file().lastModified();
    }

    public OptionalLong largestRecordTimestamp() throws IOException {
        long maxTimestampSoFar = this.maxTimestampSoFar();
        if (maxTimestampSoFar >= 0L) {
            return OptionalLong.of(maxTimestampSoFar);
        }
        return OptionalLong.empty();
    }

    public long largestTimestamp() throws IOException {
        long maxTimestampSoFar = this.maxTimestampSoFar();
        if (maxTimestampSoFar >= 0L) {
            return maxTimestampSoFar;
        }
        return this.lastModified();
    }

    public void setLastModified(long ms) throws IOException {
        FileTime fileTime = FileTime.fromMillis(ms);
        Files.setLastModifiedTime(this.log.file().toPath(), fileTime);
        Files.setLastModifiedTime(this.offsetIndexFile().toPath(), fileTime);
        Files.setLastModifiedTime(this.timeIndexFile().toPath(), fileTime);
    }

    public static LogSegment open(File dir, long baseOffset, LogConfig config, Time time, int initFileSize, boolean preallocate) throws IOException {
        return LogSegment.open(dir, baseOffset, config, time, false, initFileSize, preallocate, "");
    }

    public static LogSegment open(File dir, long baseOffset, LogConfig config, Time time, boolean fileAlreadyExists, int initFileSize, boolean preallocate, String fileSuffix) throws IOException {
        int maxIndexSize = config.maxIndexSize;
        return new LogSegment(FileRecords.open((File)LogFileUtils.logFile(dir, baseOffset, fileSuffix), (boolean)fileAlreadyExists, (int)initFileSize, (boolean)preallocate), LazyIndex.forOffset(LogFileUtils.offsetIndexFile(dir, baseOffset, fileSuffix), baseOffset, maxIndexSize), LazyIndex.forTime(LogFileUtils.timeIndexFile(dir, baseOffset, fileSuffix), baseOffset, maxIndexSize), new TransactionIndex(baseOffset, LogFileUtils.transactionIndexFile(dir, baseOffset, fileSuffix)), baseOffset, config.indexInterval, config.randomSegmentJitter(), time);
    }

    public static void deleteIfExists(File dir, long baseOffset, String fileSuffix) throws IOException {
        LogSegment.deleteFileIfExists(LogFileUtils.offsetIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.timeIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.transactionIndexFile(dir, baseOffset, fileSuffix));
        LogSegment.deleteFileIfExists(LogFileUtils.logFile(dir, baseOffset, fileSuffix));
    }

    private static boolean deleteFileIfExists(File file) throws IOException {
        return Files.deleteIfExists(file.toPath());
    }

    static {
        FUTURE_DIR_PATTERN = Pattern.compile("^(\\S+)-(\\S+)\\.(\\S+)-future");
        KafkaMetricsGroup logFlushStatsMetricsGroup = new KafkaMetricsGroup("kafka.log", "LogFlushStats");
        LOG_FLUSH_TIMER = logFlushStatsMetricsGroup.newTimer("LogFlushRateAndTimeMs", TimeUnit.MILLISECONDS, TimeUnit.SECONDS);
    }
}

