/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.server.buffer.lab;

import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import org.apache.hadoop.util.StringUtils;
import org.apache.uniffle.server.ShuffleServerMetrics;
import org.apache.uniffle.server.buffer.lab.Chunk;
import org.apache.uniffle.server.buffer.lab.OffheapChunk;
import org.apache.uniffle.shaded.guava.base.Preconditions;
import org.apache.uniffle.shaded.guava.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ChunkCreator {
    private static final Logger LOG = LoggerFactory.getLogger(ChunkCreator.class);
    private final AtomicInteger chunkID = new AtomicInteger(1);
    private final Map<Integer, Chunk> chunkIdMap = new ConcurrentHashMap<Integer, Chunk>();
    static ChunkCreator instance;
    private final int maxAlloc;
    private final ChunkPool chunksPool;
    private final int chunkSize;

    ChunkCreator(int chunkSize, long bufferCapacity, int maxAlloc) {
        this.chunkSize = chunkSize;
        this.maxAlloc = maxAlloc;
        this.chunksPool = this.initializePool(bufferCapacity, chunkSize);
    }

    public static synchronized void initialize(int chunkSize, long bufferCapacity, int maxAlloc) {
        if (instance != null) {
            LOG.warn("ChunkCreator instance is already initialized.");
        }
        instance = new ChunkCreator(chunkSize, bufferCapacity, maxAlloc);
    }

    public static ChunkCreator getInstance() {
        return instance;
    }

    Chunk getChunk() {
        return this.getChunk(this.chunksPool.getChunkSize());
    }

    Chunk getChunk(int size) {
        Chunk chunk = null;
        ChunkPool pool = null;
        if (this.chunksPool != null && size == this.chunksPool.getChunkSize()) {
            pool = this.chunksPool;
        }
        if (pool != null && (chunk = pool.getChunk()) == null && LOG.isDebugEnabled()) {
            LOG.debug("The chunk pool is full. Reached maxCount= " + pool.getMaxCount() + ". Creating chunk outside of the pool.");
        }
        if (chunk == null) {
            chunk = this.createChunk(false, size);
        }
        chunk.init();
        return chunk;
    }

    private Chunk createChunk(boolean pool, int size) {
        int id = this.chunkID.getAndIncrement();
        if (id <= 0) {
            this.chunkID.compareAndSet(id, this.chunksPool.maxCount + 1);
            id = this.chunkID.getAndIncrement();
        }
        Preconditions.checkArgument(id > 0, "chunkId should be positive.");
        OffheapChunk chunk = new OffheapChunk(size, id, pool);
        this.chunkIdMap.put(chunk.getId(), chunk);
        ShuffleServerMetrics.counterLABChunkCreated.inc();
        return chunk;
    }

    private Chunk createChunkForPool(int chunkSize) {
        if (chunkSize != this.chunksPool.getChunkSize()) {
            return null;
        }
        return this.createChunk(true, chunkSize);
    }

    private void removeChunks(List<Integer> chunkIDs) {
        chunkIDs.forEach(this::removeChunk);
    }

    Chunk removeChunk(int chunkId) {
        Chunk c = this.chunkIdMap.remove(chunkId);
        c.getData().release();
        return c;
    }

    private ChunkPool initializePool(long bufferCapacity, int chunkSize) {
        int maxCount = Math.max((int)(bufferCapacity / (long)chunkSize), 1);
        LOG.info("Allocating ChunkPool with chunk size {}, max count {}", (Object)StringUtils.byteDesc((long)chunkSize), (Object)maxCount);
        return new ChunkPool(chunkSize, maxCount);
    }

    int getChunkSize() {
        return this.chunkSize;
    }

    int getMaxAlloc() {
        return this.maxAlloc;
    }

    synchronized void putBackChunks(List<Integer> chunks) {
        if (this.chunksPool == null) {
            this.removeChunks(chunks);
            return;
        }
        for (int chunkID : chunks) {
            Chunk chunk = this.chunkIdMap.get(chunkID);
            if (chunk != null) {
                if (chunk.isFromPool()) {
                    this.chunksPool.putbackChunks(chunk);
                    continue;
                }
                this.removeChunk(chunkID);
                continue;
            }
            LOG.warn("Chunk {} can not be found in chunkIdMap, ignore it", (Object)chunkID);
        }
        ShuffleServerMetrics.gaugeLABChunkReclaimed.set((double)this.chunksPool.reclaimedChunks.size());
        ShuffleServerMetrics.gaugeLABChunkPoolRemainPercent.set((double)this.chunksPool.reclaimedChunks.size() * 100.0 / (double)this.chunksPool.chunkCount.get());
    }

    private class ChunkPool {
        private final int chunkSize;
        private final int maxCount;
        private final BlockingQueue<Chunk> reclaimedChunks;
        private final ScheduledExecutorService scheduleThreadPool;
        private static final int statThreadPeriod = 300;
        private final AtomicLong chunkCount = new AtomicLong();
        private final LongAdder reusedChunkCount = new LongAdder();

        ChunkPool(int chunkSize, int maxCount) {
            this.chunkSize = chunkSize;
            this.maxCount = maxCount;
            this.reclaimedChunks = new LinkedBlockingQueue<Chunk>();
            String n = Thread.currentThread().getName();
            this.scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(n + "-ChunkPool Statistics").setDaemon(true).build());
            this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(), 300L, 300L, TimeUnit.SECONDS);
        }

        Chunk getChunk() {
            Chunk chunk = (Chunk)this.reclaimedChunks.poll();
            if (chunk != null) {
                chunk.reset();
                this.reusedChunkCount.increment();
                ShuffleServerMetrics.gaugeLABChunkReclaimed.set((double)this.reclaimedChunks.size());
                ShuffleServerMetrics.gaugeLABChunkPoolRemainPercent.set((double)this.reclaimedChunks.size() * 100.0 / (double)this.chunkCount.get());
                ShuffleServerMetrics.counterLABChunkReused.inc();
            } else {
                long created;
                while ((created = this.chunkCount.get()) < (long)this.maxCount) {
                    if (!this.chunkCount.compareAndSet(created, created + 1L)) continue;
                    chunk = ChunkCreator.this.createChunkForPool(this.chunkSize);
                    break;
                }
            }
            return chunk;
        }

        int getChunkSize() {
            return this.chunkSize;
        }

        private void putbackChunks(Chunk c) {
            int toAdd = this.maxCount - this.reclaimedChunks.size();
            if (c.isFromPool() && c.size == this.chunkSize && toAdd > 0) {
                this.reclaimedChunks.add(c);
            } else {
                ChunkCreator.this.removeChunk(c.getId());
            }
        }

        private int getMaxCount() {
            return this.maxCount;
        }

        private class StatisticsThread
        extends Thread {
            StatisticsThread() {
                super("MemStoreChunkPool.StatisticsThread");
                this.setDaemon(true);
            }

            @Override
            public void run() {
                this.logStats();
            }

            private void logStats() {
                long created = ChunkPool.this.chunkCount.get();
                long reused = ChunkPool.this.reusedChunkCount.sum();
                long total = created + reused;
                LOG.info("ChunkPool stats (chunk size={}): current pool size={}, created chunk count={}, reused chunk count={}, reuseRatio={}", new Object[]{ChunkPool.this.chunkSize, ChunkPool.this.reclaimedChunks.size(), created, reused, total == 0L ? "0" : StringUtils.formatPercent((double)((float)reused / (float)total), (int)2)});
            }
        }
    }
}

